package com.ldodds.sparql;

import java.awt.*;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

import javax.swing.*;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.undo.UndoManager;

import com.ldodds.util.FileFilters;

public class TwinkleFrame extends JFrame
{
    private JTextArea _query;
    private JTextArea _results;
    private JTextField _dataUrl;
    private JTextField _baseUrl;
    private UndoManager _undoManager;
    private JLabel _statusBar;
    private JPanel _resultsPanel;    
    private JTable _resultsTable;
    
    public TwinkleFrame(String title)
    {
        super(title);
        
        Twinkle.ui = this;
        _query = new JTextArea();
        _results = new JTextArea();        
        _dataUrl = new JTextField(24);
        _dataUrl.setToolTipText("Local Data File or URL");
        _baseUrl = new JTextField(24);
        _baseUrl.setToolTipText("Base URL for resolving relative URLs");
        _statusBar = new JLabel("");
        _undoManager = new UndoManager();
        _resultsTable = new JTable();
                
        setDefaultWindowSize();

		addWindowListener();		          

        createMenu();        
		
        createPanels();
                                
    }

    /**
     * 
     */
    private void addWindowListener()
    {
        addWindowListener(new WindowAdapter() {
        	public void windowClosing(WindowEvent e) 
        	{        		
        		System.exit(0);
			}
    	});
    }

    /**
     * 
     */
    private void setDefaultWindowSize()
    {
        int inset = 75;
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        setBounds(inset, inset, 
                  screenSize.width - inset*2, 
                  screenSize.height-inset*2);
    }

    /**
     * 
     */
    private void createPanels()
    {
        JSplitPane splitter = new JSplitPane(
                JSplitPane.VERTICAL_SPLIT,
                createQueryPanel(),
                createResultsPanel());
        
        getContentPane().add(splitter);
        
    }

    private JPanel createQueryPanel()
    {
        JPanel queryPanel = new JPanel();
        queryPanel.setLayout(new BorderLayout());
        
        _query.getDocument().addUndoableEditListener(_undoManager);       
        JScrollPane scrollPane = new JScrollPane(_query);
        queryPanel.add( scrollPane, BorderLayout.CENTER );
        queryPanel.add( createOptionsPanel(), BorderLayout.EAST);
        
        return queryPanel;
    }
    
    private JPanel createOptionsPanel()
    {
        JPanel options = new JPanel();
        
        GridBagLayout layout = new GridBagLayout();
        options.setLayout(layout);
        
        JLabel baseUrlLabel = new JLabel("Base URI");
        baseUrlLabel.setLabelFor(_baseUrl);
        
        GridBagConstraints baseLabelConstraints = new GridBagConstraints();
        baseLabelConstraints.gridx=0;
        baseLabelConstraints.gridy=0;
        baseLabelConstraints.gridwidth=3;
        baseLabelConstraints.gridheight=1;
        baseLabelConstraints.anchor = GridBagConstraints.NORTHWEST;
        baseLabelConstraints.insets = new Insets(0, 10, 0, 12);
        layout.setConstraints(baseUrlLabel, baseLabelConstraints);
        options.add(baseUrlLabel);
        
        GridBagConstraints baseFieldConstraints = new GridBagConstraints();
        baseFieldConstraints.anchor = GridBagConstraints.NORTHWEST;
        baseFieldConstraints.gridx=3;
        baseFieldConstraints.gridy=0;
        baseFieldConstraints.gridwidth=4;
        baseFieldConstraints.gridheight=1;
        baseFieldConstraints.insets = new Insets(0, 10, 0, 6);
        layout.setConstraints(_baseUrl, baseFieldConstraints);
        options.add(_baseUrl);
        
        JLabel dataUrlLabel = new JLabel("Data URL");
        dataUrlLabel.setLabelFor(_dataUrl);
        
        GridBagConstraints dataUrlLabelConstraints = new GridBagConstraints();
        dataUrlLabelConstraints.anchor = GridBagConstraints.NORTHWEST;
        dataUrlLabelConstraints.gridx=0;
        dataUrlLabelConstraints.gridy=1;
        dataUrlLabelConstraints.gridwidth=3;
        dataUrlLabelConstraints.gridheight=1;
        dataUrlLabelConstraints.insets = new Insets(0, 10, 0, 6);
        layout.setConstraints(dataUrlLabel, dataUrlLabelConstraints);
        options.add(dataUrlLabel);

        GridBagConstraints dataUrlConstraints = new GridBagConstraints();
        dataUrlConstraints.anchor = GridBagConstraints.NORTHWEST;
        dataUrlConstraints.gridx=3;
        dataUrlConstraints.gridy=1;
        dataUrlConstraints.gridwidth=3;
        dataUrlConstraints.gridheight=1;
        dataUrlConstraints.insets = new Insets(0, 10, 0, 6);
        layout.setConstraints(_dataUrl, dataUrlConstraints);
        options.add(_dataUrl);
        
        JButton chooseDataButton = createChooseDataButton();
        GridBagConstraints chooseDataButtonConstraints = new GridBagConstraints();
        chooseDataButtonConstraints.anchor = GridBagConstraints.NORTHWEST;
        chooseDataButtonConstraints.gridx=10;
        chooseDataButtonConstraints.gridy=1;
        chooseDataButtonConstraints.gridwidth=3;
        chooseDataButtonConstraints.gridheight=1;
        chooseDataButtonConstraints.insets = new Insets(0, 10, 0, 6);
        layout.setConstraints(chooseDataButton, chooseDataButtonConstraints);
        options.add(chooseDataButton);

        
        
        GridBagConstraints runButtonConstraints = new GridBagConstraints();
        JButton executeButton = createRunQueryButton();
        runButtonConstraints.gridx=2;
        runButtonConstraints.gridy=4;
        runButtonConstraints.gridwidth=2;
        runButtonConstraints.gridheight=1;
        runButtonConstraints.anchor = GridBagConstraints.SOUTHEAST;
        runButtonConstraints.insets = new Insets(5, 10, 0, 6);
        layout.setConstraints(executeButton, runButtonConstraints);
        options.add(executeButton);        
                
        return options;
    }
    
    private JPanel createResultsPanel()
    {
        _resultsPanel = new JPanel(new CardLayout());
        
        
        //text "card"
        JPanel results = new JPanel(new BorderLayout());       
        //TODO should use JEditorPane?
        _results.setEditable(false);        
        JScrollPane scrollPane = new JScrollPane(_results);
        results.add(scrollPane, BorderLayout.CENTER);
        results.add(_statusBar, BorderLayout.SOUTH);
        _resultsPanel.add(results, "text");
        
        //table "card"
        JScrollPane tableScrollPane = new JScrollPane(_resultsTable);        
        _resultsPanel.add(tableScrollPane, "table");
        
        return _resultsPanel;
    }
    
    public void showResultsPanel(String card)
    {
        CardLayout cards = (CardLayout)_resultsPanel.getLayout();
        cards.show(_resultsPanel, card);
    }
    
    public JTextArea getQueryBox()
    {
        return _query;
    }
    
    public JTextArea getResultsBox()
    {
        return _results;
    }
    /**
     * 
     */
    private void createMenu()
    {
        JMenuBar menuBar = new JMenuBar();
        menuBar.add( MenuFactory.createFileMenu() );     
        menuBar.add( MenuFactory.createEditMenu() );
        menuBar.add( MenuFactory.createSourceMenu() );
        menuBar.add( MenuFactory.createResultsMenu() );
        setJMenuBar(menuBar);
    }
    
    private JButton createRunQueryButton()
    {
        JButton runQuery = new JButton("Run Query");
        runQuery.setToolTipText("Run the current query");
        runQuery.addActionListener(new RunQueryCommand());
        return runQuery;
    }
    
    private JButton createChooseDataButton()
    {
        JButton chooseDataButton = new JButton("File...");
        chooseDataButton.setToolTipText("Select local data file");
        chooseDataButton.addActionListener( new ActionListener()
                {
            		final JFileChooser fc = new JFileChooser(System.getProperty("user.dir"));
            		
		            public void actionPerformed(ActionEvent e)
		            {
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("txt", "Text Document") );		                
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("xml", "XML Document") );		                
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("nt", "N-Triples Document") );		                
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("ttl", "Turtle Document") );
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("n3", "N3 Document") );		                
		                fc.addChoosableFileFilter( FileFilters.getFileFilter("rdf", "RDF/XML Document") );
		                int result = fc.showDialog(Twinkle.ui, "Choose Data File");
		                if (result == JFileChooser.APPROVE_OPTION)
		                {
		                    File file = fc.getSelectedFile();
		                    Twinkle.ui.getDataUrlField().setText( file.toURI().toString() );
		                    Twinkle.showStatusBarMessage("Selected data file " + file.getAbsolutePath());
		                }
		            }
                });
        return chooseDataButton;
    }
    public JTextField getDataUrlField()
    {
        return _dataUrl;
    }
    
    /**
     * @return Returns the baseUrl.
     */
    public JTextField getBaseUrl()
    {
        return _baseUrl;
    }
    
    public UndoManager getUndoManager()
    {
        return _undoManager;
    }

    
    /**
     * @return Returns the statusBar.
     */
    public JLabel getStatusBar()
    {
        return _statusBar;
    }
    
    public JTable getResultsTable()
    {
        return _resultsTable;
    }
}
