package com.ldodds.sparql;

import java.io.*;

import com.hp.hpl.jena.util.*;
import com.hp.hpl.jena.query.*;
import com.hp.hpl.jena.query.core.ARQInternalErrorException;

import org.apache.commons.logging.*;

/**
 * 
 * @author Andy Seaborne
 * @author Leigh Dodds
 */
public class RunQuery
{
    /** No output */
    public final static String FMT_NONE = "none";

    /** Output the triples for the query */
    public final static String FMT_TUPLES = "tuples";

    /** Output a text table */
    public final static String FMT_TEXT = "text";

    /** Output an RDF graph */    
    public final static String FMT_RS_RDF = "rs/graph";

    /** Output results showing structure */
    public final static String FMT_RS_TEXT = "rs/text";

    /** Output as XML */
    public final static String FMT_RS_XML = "rs/xml";

    private Log _log = LogFactory.getLog(RunQuery.class);
    private OutputStream _out;
    private QueryContext _context;

    public RunQuery(QueryContext context)
    {
        this(System.out, context);
    }

    public RunQuery(OutputStream out, QueryContext context)
    {
        _out = out;
        _context = context;
    }

    /** 
     * Execute a query given some information.
     * Set what you know in the QueryCntext object - this code will
     * sort out missing stuff if it can.
     */
    public void query() throws QueryException
    {
        addDataSourceToContext();

        createQuery();
        
        //never null that we throw an exception
        //if (_context.query == null) return;

        bindData();

        // Check there is a dataset
        if (_context.query.getDataSet() == null
                && _context.query.getSourceURIs() == null
                && _context.query.getNamedSourceURIs() == null)

        {
            throw new QueryException("No dataset specified");
        }
        execute();
    }

    private void addDataSourceToContext() throws QueryException
    {
        try
        {
            if (_context.fileManager == null) _context.fileManager = FileManager.get();
            if (_context.dataURL != null)
                _context.model = _context.fileManager.loadModel(_context.dataURL,
                        _context.baseURI, _context.dataSyntax);
        }
        catch (Exception ex)
        {
            ex.printStackTrace();
            throw new QueryException("Error loading URL", ex);
        }
    }

    /**
     * Reparse query if necessary, and associate baseURI.
     */
    void createQuery() throws QueryException
    {
        // Create the query
        if (_context.query == null)
        {
            _context.query = parseQuery(_context.queryString, _context.syntax);
        }
        if (_context.baseURI != null) 
        {
            _context.query.setBaseURI(_context.baseURI);
        }
    }

    private void bindData()
    {
        // Bind up the data
        if (_context.model != null) _context.query.setDataSet(_context.model);
    }

    private void execute() throws QueryException
    {
        try
        {
            // Execute
            QueryExecution qe = QueryFactory.createQueryExecution(_context.query);
            if (_context.fileManager != null)
            {
                qe.setFileManager(_context.fileManager);
            }
            qe.init();

            QueryRunner runner = AbstractQueryRunner.getQueryRunner(_context.query, _out);
            runner.doQuery(_context, qe);        
            
            qe.close();
        }
        catch (ARQInternalErrorException intEx)
        {
            //TEMPORARY
            if (intEx.getCause() != null)
            {
                System.err.println("Cause:");
                intEx.getCause().printStackTrace(System.err);
                System.err.println();
            }
            intEx.printStackTrace(System.err);
            throw intEx;
        }
        catch (QueryException qEx)
        {
            qEx.printStackTrace();
            throw qEx;
        }
        catch (Exception ex)
        {
            ex.printStackTrace();
            throw new QueryException("Unexpected error executing query", ex);
        }
    }

    private Query parseQuery(String queryString, String syntax) throws QueryException
    {
        try
        {
            Query query = Query.create(queryString, syntax);
            return query;
        }
        catch (ARQInternalErrorException intEx)
        {
            //TEMPORARY
            System.err.println(intEx.getMessage());
            if (intEx.getCause() != null)
            {
                System.err.println("Cause:");
                intEx.getCause().printStackTrace(System.err);
                System.err.println();
            }
            intEx.printStackTrace(System.err);
            throw intEx;
        }
        catch (QueryException qEx)
        {
            qEx.printStackTrace();
            throw qEx;
        }
        catch (Exception ex)
        {
            ex.printStackTrace();
            throw new QueryException("Unexpected error parsing query", ex);
        }
    }

}