package com.ldodds.sparql;

import java.awt.Toolkit;
import java.awt.event.*;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import java.io.*;

import javax.swing.*;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.text.DefaultEditorKit;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;

import com.hp.hpl.jena.util.FileManager;
import com.hp.hpl.jena.util.FileUtils;

import com.ldodds.util.FileFilters;

/**
 * @author ldodds
 */
public class MenuFactory
{
    private static int META_KEY = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    
	private static final class OpenQueryAction extends AbstractAction
    {
        private JFileChooser fc;
        
        public OpenQueryAction()
        {
            //default to users current directory as default
            fc = new JFileChooser( System.getProperty("user.dir") );
            fc.setDialogTitle("Open Query");
            fc.addChoosableFileFilter( FileFilters.getFileFilter("txt", "Text Files"));
            fc.addChoosableFileFilter( FileFilters.getFileFilter("rq", "Sparql Queries"));            
        }
        
        public void actionPerformed(ActionEvent e) 
        {
            int result = fc.showOpenDialog(Twinkle.ui);
            if (result == JFileChooser.APPROVE_OPTION)
            {
                File queryFile = fc.getSelectedFile();
                try
                {
                    FileInputStream fin = new FileInputStream(queryFile);
                    String query = FileManager.get().readWholeFileAsUTF8(fin) ;
                    Twinkle.ui.getQueryBox().setText(query);
                    Twinkle.showStatusBarMessage("Loaded Query " + queryFile.getAbsolutePath());
                } catch (FileNotFoundException fne)
                {
                    
                }
            }
        }
    }

	private abstract static class SaveAction extends AbstractAction
	{
	    private JFileChooser fc;

	    public SaveAction()
	    {
	        fc = new JFileChooser(System.getProperty("user.dir"));
	    }
        public void actionPerformed(ActionEvent e)
        {
            fc.setDialogTitle(getDialogTitle());
            addFileFilters(fc);
            int result = fc.showSaveDialog(Twinkle.ui);
            if (result == JFileChooser.APPROVE_OPTION)
            {
                File queryFile = fc.getSelectedFile();
                try
                {
                    FileOutputStream fout = new FileOutputStream(queryFile);
                    OutputStreamWriter out = new OutputStreamWriter(fout, "UTF-8");
                    out.write(getTextToSave());
                    out.flush();
                    out.close();
                    Twinkle.showStatusBarMessage("Saved to " + queryFile.getAbsolutePath());
                } catch (IOException ioe)
                {
                    ioe.printStackTrace();
                }
            }
        }
	    
	    abstract String getTextToSave();
	    abstract String getDialogTitle();
	    abstract void addFileFilters(JFileChooser chooser);
	}
	
	private static final class SaveQueryAction extends SaveAction
	{
	    String getTextToSave()
	    {
	        return Twinkle.ui.getQueryBox().getText();
	    }
	    
	    String getDialogTitle()
	    {
	        return "Save Query";
	    }
	    
	    void addFileFilters(JFileChooser fc)
	    {
            fc.addChoosableFileFilter( FileFilters.getFileFilter("txt", "Text Files"));	        
	        fc.addChoosableFileFilter( FileFilters.getFileFilter("rq", "Sparql Queries"));
	    }
	}
	
	private static final class SaveResultsAction extends SaveAction
	{
	    String getTextToSave()
	    {
	        return Twinkle.ui.getResultsBox().getText();
	    }
	    
	    String getDialogTitle()
	    {
	        return "Save Results";
	    }
	    
	    void addFileFilters(JFileChooser fc)
	    {
            fc.addChoosableFileFilter( FileFilters.getFileFilter("txt", "Text Files"));	        
            fc.addChoosableFileFilter( FileFilters.getFileFilter("xml", "XML Documents"));	        
            fc.addChoosableFileFilter( FileFilters.getFileFilter("rdf", "RDF/XML Documents"));            
	    }
	    
	}
	
    public static JMenu createFileMenu()
	{
		//File menu
        JMenu menu = new JMenu("File");
        menu.setMnemonic(KeyEvent.VK_F);
        
        JMenuItem openMenuItem = new JMenuItem("Open Query");
        openMenuItem.setMnemonic(KeyEvent.VK_O);
        openMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, META_KEY));
        openMenuItem.addActionListener(new OpenQueryAction());
        menu.add(openMenuItem);
        
        JMenuItem saveMenuItem = new JMenuItem("Save Query");
        saveMenuItem.setMnemonic(KeyEvent.VK_S);
        saveMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, META_KEY));
        saveMenuItem.addActionListener(new SaveQueryAction());
        menu.add(saveMenuItem);

        JMenuItem saveResultsMenuItem = new JMenuItem("Save Results");
        saveResultsMenuItem.setMnemonic(KeyEvent.VK_A);
        saveResultsMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, META_KEY));
        saveResultsMenuItem.addActionListener(new SaveResultsAction());
        menu.add(saveResultsMenuItem);
                
        JMenuItem exitMenuItem = new JMenuItem("Exit");
        exitMenuItem.setMnemonic(KeyEvent.VK_X);
        exitMenuItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) 
            {
                System.exit(0);
            }
        });
        
        menu.add(exitMenuItem);	
        return menu;	
	}

	public static JMenu createEditMenu()
	{
	    JMenu menu = new JMenu("Edit");
	    menu.setMnemonic(KeyEvent.VK_E);
	    
	    JMenuItem undoMenuItem = new JMenuItem("Undo");
	    undoMenuItem.setMnemonic(KeyEvent.VK_U);
	    undoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, META_KEY));
	    undoMenuItem.addActionListener(new ActionListener()
	            {
		            public void actionPerformed(ActionEvent e)
		            {
		                try
		                {
		                    Twinkle.ui.getUndoManager().undo();
		                } catch (CannotUndoException ex)
		                {
		                    Twinkle.ui.getToolkit().beep();
		                }
		            }
	            });
	    
	    menu.add(undoMenuItem);
	    
	    JMenuItem redoMenuItem = new JMenuItem("Redo");
	    redoMenuItem.setMnemonic(KeyEvent.VK_R);
	    redoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, META_KEY));
	    redoMenuItem.addActionListener(new ActionListener()
	            {
		            public void actionPerformed(ActionEvent e)
		            {
		                try
		                {
		                    Twinkle.ui.getUndoManager().redo();
		                } catch (CannotRedoException ex)
		                {
		                    Twinkle.ui.getToolkit().beep();
		                }
		            }
	            });
	    
	    menu.add(redoMenuItem);	    

	    JMenuItem cutMenuItem = new JMenuItem("Cut");
	    cutMenuItem.setMnemonic(KeyEvent.VK_X);
	    cutMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, META_KEY));

	    cutMenuItem.addActionListener(
	            Twinkle.ui.getQueryBox().getActionMap().get(DefaultEditorKit.cutAction));
	    menu.add(cutMenuItem);
	    
	    JMenuItem copyMenuItem = new JMenuItem("Copy");
	    copyMenuItem.setMnemonic(KeyEvent.VK_C);
	    copyMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, META_KEY));
	    copyMenuItem.addActionListener(
	            Twinkle.ui.getQueryBox().getActionMap().get(DefaultEditorKit.copyAction));
	    menu.add(copyMenuItem);
	    
	    JMenuItem pasteMenuItem = new JMenuItem("Paste");
	    pasteMenuItem.setMnemonic(KeyEvent.VK_V);
	    pasteMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, META_KEY));
	    pasteMenuItem.addActionListener(
	            Twinkle.ui.getQueryBox().getActionMap().get(DefaultEditorKit.pasteAction));
	    menu.add(pasteMenuItem);
	    
	    return menu;
	}
	public static JMenu createSourceMenu()
	{
	    JMenu menu = new JMenu("Source");
	    menu.setMnemonic(KeyEvent.VK_S);
	    
	    ButtonGroup group = new ButtonGroup();
	    JMenuItem rdfXML = new JRadioButtonMenuItem("RDF/XML");
	    rdfXML.setActionCommand(FileUtils.langXML);
	    rdfXML.setSelected(true);
	    group.add(rdfXML);
	    menu.add(rdfXML);

	    JMenuItem ntriples = new JRadioButtonMenuItem("N-Triples");
	    ntriples.setActionCommand(FileUtils.langNTriple);
	    group.add(ntriples);
	    menu.add(ntriples);

	    JMenuItem turtle = new JRadioButtonMenuItem("Turtle");
	    turtle.setActionCommand(FileUtils.langTurtle);
	    group.add(turtle);
	    menu.add(turtle);

	    JMenuItem nthree = new JRadioButtonMenuItem("N3");
	    nthree.setActionCommand(FileUtils.langN3);
	    group.add(nthree);
	    menu.add(nthree);
	    
	    ActionListener listener = new ActionListener() {
				public void actionPerformed(ActionEvent e)
				{				    
				    Twinkle.context.dataSyntax = e.getActionCommand();
				    Twinkle.showStatusBarMessage("Data syntax set to " + e.getActionCommand());
				}	        
	    	};
	    
	    rdfXML.addActionListener(listener);
	    ntriples.addActionListener(listener);
	    turtle.addActionListener(listener);
	    nthree.addActionListener(listener);
	    return menu; 
	}
	
	public static JMenu createResultsMenu()
	{
	    JMenu menu = new JMenu("Results");
	    menu.setMnemonic(KeyEvent.VK_R);

	    ButtonGroup group = new ButtonGroup();
	    JMenuItem text = new JRadioButtonMenuItem("Text");
	    text.setActionCommand(RunQuery.FMT_TEXT);
	    text.setSelected(true);
	    group.add(text);
	    menu.add(text);
	    
	    JMenuItem xml = new JRadioButtonMenuItem("XML");
	    xml.setActionCommand(RunQuery.FMT_RS_XML);
	    group.add(xml);
	    menu.add(xml);
	   	    	    
	    JMenuItem table = new JRadioButtonMenuItem("Table");
	    //TODO add constant
	    table.setActionCommand("table");
	    group.add(table);
	    menu.add(table);
	    
	    //TODO Add the rest of the result formats
	    
	    ActionListener listener = new ActionListener() {
			public void actionPerformed(ActionEvent e)
			{
			    Twinkle.context.outputFormat = e.getActionCommand();
			    Twinkle.showStatusBarMessage("Results format set to " + e.getActionCommand());
			    //TODO tidy this up
			    if ("table".equals(e.getActionCommand()))
			    {
			        Twinkle.ui.showResultsPanel("table");			        
			    }
			    else
			    {
			        Twinkle.ui.showResultsPanel("text");
			    }
			}	        
    	};
	    text.addActionListener(listener);
	    xml.addActionListener(listener);
	    table.addActionListener(listener);
	    
	    return menu;
	}
	
}
